%   #######################################################################
%   Implementing Finite Difference method for contingent claim pricing.
%   Created by: Nik Tuzov, www.ntuzov.com
%   This program creates and saves dataset "plotdata.mat" which is later used
%   in GUI for plotting.
%   #######################################################################
 
%   #######################################################################
%   Example1: a standard European call. Initial data:
    
     S = 110;            % stock price, current
     K = 100;            % strike price
     r = 0.05;           % risk - free rate
     sigma = 0.3;        % volatility
     T = 1;              % time to maturity     
     theta = 0.5;        % theta = 1: fully implicit method
                         % theta = 0: fully explicit method
                         % theta = 0.5: Crank and Nicholson scheme     
 %   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
     % True call price, Black-Scholes-based: 
       [BS_call, BS_put]  = blsprice(S, K, r, T, sigma);
    
      % L, state space localization parameter:      
      L = max( abs(log(S) + (r - sigma^2 /2) + 4 * sigma * sqrt(T)), abs(log(S) + (r - sigma^2 /2) - 4 * sigma * sqrt(T)) );     
      
      max_space =  2000; % maximal number of space nodes  
      max_time = 200;    % maximal number of time nodes 
      
      % steps for the number of time and space nodes:
      step_time = 10;     
      step_space = 20;      
      
      % grid for time and space
      [space, time] = meshgrid([step_space : step_space : max_space], [step_time : step_time : max_time]);   
      
      % calculating the pricing error:
      resid = space;            
      for i = 1 : size(resid,1)
          for j = 1 : size(resid, 2)
              resid(i,j) = abs( BS_call - claim(@call, 'euro', S, r, sigma, T, L, space(i, j), time(i, j), theta));          
          end;
      end;              
      
      % Save data:
      data.space = space;
      data.time = time;
      data.vanilla_euro_call = resid;           

%  #######################################################################  
%   Example2: an Exotic European option. Initial data are the same as for
%   Example1.
%   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++      
      % calculating the price estimate:
      theta = 0.5;
      price = space;            
      for i = 1 : size(price,1)
          for j = 1 : size(price, 2)
              price(i,j) = claim(@root, 'euro', S, r, sigma, T, L, space(i, j), time(i, j), theta);                
          end;
      end;      
      % Save data:
      data.exotic_euro_call = price;  
           
%  #######################################################################
%   Example3: an standard American put. Initial data are the same 
%   as for Example1     
%   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++     
     theta = 1;    
     % calculating the price estimate:
      price = space;            
      for i = 1 : size(price,1)
          for j = 1 : size(price, 2)
              price(i,j) = claim(@put, 'amer', S, r, sigma, T, L, space(i, j), time(i, j), theta);                           
          end;
      end;
      % Save data:
      data.vanilla_amer_put = price;       
 
% %  ####################################################################### 
% %   Example4: an exotic  American "sine" option. Initial data are the same 
%     as for Example1     
% %   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++     
     theta = 1;    
     % calculating the price estimate:
      price = space;            
      for i = 1 : size(price,1)
          for j = 1 : size(price, 2)
              price(i,j) = claim(@sine_opt, 'amer', S, r, sigma, T, L, space(i, j), time(i, j), theta);                            
          end;
      end;     
       % Save data:
      data.exotic_amer_put = price;       
% %  ####################################################################### 
      save('plotdata.mat', 'data');
          
      


      
    